/**
 * Truth Protocol - API Client
 * Connects to Human Router backend for consensus analysis
 *
 * WeRAI / HumAIn Integration
 * Patent Pending: USPTO #63900179
 */

const API_BASE = 'https://werai.ca/api/truth';

// Local cache for quick lookups
const entityCache = new Map();
const CACHE_TTL = 5 * 60 * 1000; // 5 minutes

/**
 * Quick check for cached entity data
 */
async function quickCheck(name, context = '') {
  const cacheKey = `${name}:${context}`.toLowerCase();

  // Check local cache first
  const cached = entityCache.get(cacheKey);
  if (cached && (Date.now() - cached.timestamp) < CACHE_TTL) {
    return cached.data;
  }

  try {
    const response = await fetch(`${API_BASE}/quick-check`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        subject: name,
        context: context,
        mode: 'cached'
      })
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    const data = await response.json();

    // Cache the result
    entityCache.set(cacheKey, {
      data: data,
      timestamp: Date.now()
    });

    return data;
  } catch (error) {
    console.error('Truth Protocol API error:', error);
    return { status: 'error', error: error.message };
  }
}

/**
 * Get full analysis for an entity and topic
 */
async function getFullAnalysis(subject, topic, historicalStatements = [], currentStatements = []) {
  try {
    const response = await fetch(`${API_BASE}/analyze`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        subject: subject,
        topic: topic,
        historical_statements: historicalStatements,
        current_statements: currentStatements
      })
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Truth Protocol full analysis error:', error);
    return { status: 'error', error: error.message };
  }
}

/**
 * Get entity summary (all analyses for one entity)
 */
async function getEntitySummary(name) {
  try {
    const response = await fetch(`${API_BASE}/entity/${encodeURIComponent(name)}/summary`);

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Truth Protocol entity summary error:', error);
    return { status: 'error', error: error.message };
  }
}

/**
 * Check API health
 */
async function checkHealth() {
  try {
    const response = await fetch(`${API_BASE}/health`);
    return await response.json();
  } catch (error) {
    return { status: 'offline', error: error.message };
  }
}

/**
 * Clear local cache
 */
function clearCache() {
  entityCache.clear();
}

/**
 * Get cache stats
 */
function getCacheStats() {
  return {
    size: entityCache.size,
    entries: Array.from(entityCache.keys())
  };
}

// Export for use in other modules
window.TruthProtocol = window.TruthProtocol || {};
window.TruthProtocol.api = {
  quickCheck,
  getFullAnalysis,
  getEntitySummary,
  checkHealth,
  clearCache,
  getCacheStats,
  API_BASE
};
