/**
 * Truth Protocol - Service Worker
 * Background processing for the extension
 *
 * WeRAI / HumAIn Integration
 * Patent Pending: USPTO #63900179
 */

const API_BASE = 'https://werai.ca/api/truth';

// Initialize on install
chrome.runtime.onInstalled.addListener(() => {
  console.log('Truth Protocol extension installed');

  // Set default settings
  chrome.storage.local.set({
    detectionEnabled: true,
    indicatorsEnabled: true,
    apiBase: API_BASE
  });
});

// Listen for messages from content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'checkEntity') {
    checkEntity(message.name, message.context)
      .then(sendResponse)
      .catch(error => sendResponse({ error: error.message }));
    return true; // Keep channel open for async response
  }

  if (message.action === 'getFullAnalysis') {
    getFullAnalysis(message.subject, message.topic)
      .then(sendResponse)
      .catch(error => sendResponse({ error: error.message }));
    return true;
  }

  if (message.action === 'checkHealth') {
    checkHealth()
      .then(sendResponse)
      .catch(error => sendResponse({ error: error.message }));
    return true;
  }
});

// Open side panel when extension icon is clicked (alternative action)
chrome.action.onClicked.addListener((tab) => {
  chrome.sidePanel.open({ tabId: tab.id });
});

/**
 * Quick check for entity status
 */
async function checkEntity(name, context = '') {
  try {
    const response = await fetch(`${API_BASE}/quick-check`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        subject: name,
        context: context,
        mode: 'cached'
      })
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error checking entity:', error);
    return { status: 'error', error: error.message };
  }
}

/**
 * Get full analysis
 */
async function getFullAnalysis(subject, topic) {
  try {
    const response = await fetch(`${API_BASE}/analyze`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        subject: subject,
        topic: topic
      })
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error getting full analysis:', error);
    return { status: 'error', error: error.message };
  }
}

/**
 * Check API health
 */
async function checkHealth() {
  try {
    const response = await fetch(`${API_BASE}/health`);
    return await response.json();
  } catch (error) {
    return { status: 'offline', error: error.message };
  }
}

// Badge update based on page entities
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url) {
    // Could update badge with entity count here
  }
});
