/**
 * Truth Protocol - Annotator
 * Injects inline annotations next to entity names
 *
 * WeRAI / HumAIn Integration
 * Patent Pending: USPTO #63900179
 */

// Status colors
const STATUS_COLORS = {
  consistent: '#22c55e',    // Green
  evolved: '#eab308',       // Yellow
  contradicted: '#f97316',  // Orange
  memory_hole: '#ef4444',   // Red
  unknown: '#6b7280',       // Gray
  loading: '#3b82f6',       // Blue
  error: '#dc2626'          // Dark red
};

// Status icons (simple unicode)
const STATUS_ICONS = {
  consistent: '\u2713',     // Check mark
  evolved: '\u21c4',        // Arrows
  contradicted: '\u26a0',   // Warning
  memory_hole: '\u2718',    // X mark
  unknown: '\u003f',        // Question mark
  loading: '\u23f3',        // Hourglass
  error: '\u2716'           // X
};

/**
 * Create annotation element for an entity
 */
function createAnnotation(entityName, status = 'unknown') {
  const indicator = document.createElement('span');
  indicator.className = 'tp-indicator';
  indicator.dataset.entity = entityName;
  indicator.dataset.status = status;

  const dot = document.createElement('span');
  dot.className = 'tp-dot';
  dot.style.backgroundColor = STATUS_COLORS[status] || STATUS_COLORS.unknown;
  dot.textContent = STATUS_ICONS[status] || STATUS_ICONS.unknown;

  indicator.appendChild(dot);

  // Add click handler
  indicator.addEventListener('click', (e) => {
    e.preventDefault();
    e.stopPropagation();
    showEntityOverlay(entityName, indicator);
  });

  // Add hover handler
  indicator.addEventListener('mouseenter', () => {
    showHoverCard(entityName, indicator);
  });

  indicator.addEventListener('mouseleave', () => {
    hideHoverCard();
  });

  return indicator;
}

/**
 * Annotate a text node with entity indicators
 */
function annotateTextNode(textNode, entities) {
  if (!textNode.parentElement) return;

  const text = textNode.textContent;
  const parent = textNode.parentElement;

  // Create document fragment for replacement
  const fragment = document.createDocumentFragment();
  let lastIndex = 0;

  for (const entity of entities) {
    // Add text before this entity
    if (entity.index > lastIndex) {
      fragment.appendChild(
        document.createTextNode(text.slice(lastIndex, entity.index))
      );
    }

    // Create wrapper for entity name + indicator
    const wrapper = document.createElement('span');
    wrapper.className = 'tp-entity-wrapper tp-annotated';
    wrapper.dataset.entity = entity.name;

    // Entity name text
    wrapper.appendChild(
      document.createTextNode(entity.matchedText)
    );

    // Add indicator
    const indicator = createAnnotation(entity.name);
    wrapper.appendChild(indicator);

    fragment.appendChild(wrapper);

    lastIndex = entity.index + entity.length;

    // Queue API check for this entity
    queueEntityCheck(entity.name);
  }

  // Add remaining text
  if (lastIndex < text.length) {
    fragment.appendChild(
      document.createTextNode(text.slice(lastIndex))
    );
  }

  // Replace the text node
  parent.replaceChild(fragment, textNode);
}

/**
 * Queue for API checks to avoid overwhelming the server
 */
const checkQueue = new Set();
let isProcessingQueue = false;

function queueEntityCheck(entityName) {
  checkQueue.add(entityName);
  processCheckQueue();
}

async function processCheckQueue() {
  if (isProcessingQueue || checkQueue.size === 0) return;

  isProcessingQueue = true;

  while (checkQueue.size > 0) {
    const entityName = checkQueue.values().next().value;
    checkQueue.delete(entityName);

    try {
      await checkEntityStatus(entityName);
    } catch (error) {
      console.error('Error checking entity:', entityName, error);
    }

    // Small delay between checks
    await new Promise(resolve => setTimeout(resolve, 100));
  }

  isProcessingQueue = false;
}

/**
 * Check entity status via API
 */
async function checkEntityStatus(entityName) {
  // Update indicators to loading state
  updateEntityIndicators(entityName, 'loading');

  try {
    const result = await window.TruthProtocol.api.quickCheck(entityName);

    if (result.status === 'error') {
      updateEntityIndicators(entityName, 'error');
    } else if (result.status === 'unknown' || result.queued) {
      updateEntityIndicators(entityName, 'unknown');
    } else {
      // Determine status from result
      const status = determineStatus(result);
      updateEntityIndicators(entityName, status, result);
    }
  } catch (error) {
    updateEntityIndicators(entityName, 'error');
  }
}

/**
 * Determine status from API result
 */
function determineStatus(result) {
  if (result.classification) {
    const classification = result.classification.toLowerCase();
    if (classification.includes('consistent')) return 'consistent';
    if (classification.includes('evolved')) return 'evolved';
    if (classification.includes('memory_hole')) return 'memory_hole';
    if (classification.includes('contradict')) return 'contradicted';
  }
  return 'unknown';
}

/**
 * Update all indicators for an entity
 */
function updateEntityIndicators(entityName, status, data = null) {
  const indicators = document.querySelectorAll(
    `.tp-indicator[data-entity="${entityName}"]`
  );

  indicators.forEach(indicator => {
    indicator.dataset.status = status;
    if (data) {
      indicator.dataset.result = JSON.stringify(data);
    }

    const dot = indicator.querySelector('.tp-dot');
    if (dot) {
      dot.style.backgroundColor = STATUS_COLORS[status] || STATUS_COLORS.unknown;
      dot.textContent = STATUS_ICONS[status] || STATUS_ICONS.unknown;
    }
  });
}

/**
 * Process all detected entities on the page
 */
function processDetectedEntities() {
  const detected = window.TruthProtocol.detector.scanDocument();

  for (const { node, entities } of detected) {
    try {
      annotateTextNode(node, entities);
    } catch (error) {
      console.error('Error annotating node:', error);
    }
  }

  console.log(`Truth Protocol: Annotated ${detected.length} text nodes`);
}

// Export
window.TruthProtocol = window.TruthProtocol || {};
window.TruthProtocol.annotator = {
  processDetectedEntities,
  createAnnotation,
  updateEntityIndicators,
  STATUS_COLORS,
  STATUS_ICONS
};
