/**
 * Truth Protocol - Overlay UI
 * Hover cards and detail overlays for entity information
 *
 * WeRAI / HumAIn Integration
 * Patent Pending: USPTO #63900179
 */

let activeHoverCard = null;
let activeOverlay = null;
let hoverTimeout = null;

/**
 * Show hover card for entity
 */
function showHoverCard(entityName, anchor) {
  hideHoverCard();

  hoverTimeout = setTimeout(() => {
    const card = createHoverCard(entityName);
    document.body.appendChild(card);

    // Position near anchor
    const rect = anchor.getBoundingClientRect();
    card.style.position = 'fixed';
    card.style.left = `${rect.right + 10}px`;
    card.style.top = `${rect.top}px`;

    // Adjust if off screen
    const cardRect = card.getBoundingClientRect();
    if (cardRect.right > window.innerWidth) {
      card.style.left = `${rect.left - cardRect.width - 10}px`;
    }
    if (cardRect.bottom > window.innerHeight) {
      card.style.top = `${window.innerHeight - cardRect.height - 10}px`;
    }

    activeHoverCard = card;
  }, 300);
}

/**
 * Hide hover card
 */
function hideHoverCard() {
  if (hoverTimeout) {
    clearTimeout(hoverTimeout);
    hoverTimeout = null;
  }
  if (activeHoverCard) {
    activeHoverCard.remove();
    activeHoverCard = null;
  }
}

/**
 * Create hover card element
 */
function createHoverCard(entityName) {
  const card = document.createElement('div');
  card.className = 'tp-hover-card';

  // Get cached data if available
  const indicator = document.querySelector(
    `.tp-indicator[data-entity="${entityName}"]`
  );
  const status = indicator?.dataset.status || 'unknown';
  const resultData = indicator?.dataset.result
    ? JSON.parse(indicator.dataset.result)
    : null;

  const statusColor = window.TruthProtocol.annotator.STATUS_COLORS[status];

  card.innerHTML = `
    <div class="tp-hover-header">
      <span class="tp-hover-dot" style="background: ${statusColor}"></span>
      <span class="tp-hover-name">${entityName}</span>
    </div>
    <div class="tp-hover-body">
      ${getStatusText(status, resultData)}
    </div>
    <div class="tp-hover-footer">
      Click for details
    </div>
  `;

  return card;
}

/**
 * Get status text for hover card
 */
function getStatusText(status, data) {
  switch (status) {
    case 'consistent':
      return 'Consistent public record';
    case 'evolved':
      return 'Position has evolved over time';
    case 'contradicted':
      return data?.contradiction_count
        ? `${data.contradiction_count} documented contradictions`
        : 'Contradictions detected';
    case 'memory_hole':
      return 'Previous positions denied';
    case 'loading':
      return 'Checking records...';
    case 'error':
      return 'Could not verify';
    default:
      return 'No data available';
  }
}

/**
 * Show full entity overlay
 */
function showEntityOverlay(entityName, anchor) {
  hideEntityOverlay();

  const overlay = createEntityOverlay(entityName);
  document.body.appendChild(overlay);
  activeOverlay = overlay;

  // Fetch full analysis
  loadFullAnalysis(entityName, overlay);
}

/**
 * Hide entity overlay
 */
function hideEntityOverlay() {
  if (activeOverlay) {
    activeOverlay.remove();
    activeOverlay = null;
  }
}

/**
 * Create full overlay element
 */
function createEntityOverlay(entityName) {
  const overlay = document.createElement('div');
  overlay.className = 'tp-overlay';

  const indicator = document.querySelector(
    `.tp-indicator[data-entity="${entityName}"]`
  );
  const status = indicator?.dataset.status || 'unknown';
  const statusColor = window.TruthProtocol.annotator.STATUS_COLORS[status];

  overlay.innerHTML = `
    <div class="tp-overlay-backdrop"></div>
    <div class="tp-overlay-panel">
      <div class="tp-overlay-header">
        <div class="tp-overlay-title">
          <span class="tp-overlay-dot" style="background: ${statusColor}"></span>
          TRUTH PROTOCOL: ${entityName}
        </div>
        <button class="tp-overlay-close">&times;</button>
      </div>
      <div class="tp-overlay-content">
        <div class="tp-loading">Loading analysis...</div>
      </div>
      <div class="tp-overlay-footer">
        <a href="https://werai.ca/truth-protocol.html" target="_blank" class="tp-link">
          Full Analysis on WeRAI
        </a>
        <span class="tp-powered">
          Powered by Human Router Architecture
        </span>
      </div>
    </div>
  `;

  // Close handlers
  overlay.querySelector('.tp-overlay-close').addEventListener('click', hideEntityOverlay);
  overlay.querySelector('.tp-overlay-backdrop').addEventListener('click', hideEntityOverlay);

  // Escape key
  document.addEventListener('keydown', function escHandler(e) {
    if (e.key === 'Escape') {
      hideEntityOverlay();
      document.removeEventListener('keydown', escHandler);
    }
  });

  return overlay;
}

/**
 * Load full analysis into overlay
 */
async function loadFullAnalysis(entityName, overlay) {
  const content = overlay.querySelector('.tp-overlay-content');

  try {
    // For now, use cached data if available
    const indicator = document.querySelector(
      `.tp-indicator[data-entity="${entityName}"]`
    );
    const resultData = indicator?.dataset.result
      ? JSON.parse(indicator.dataset.result)
      : null;

    if (resultData && resultData.consensus) {
      content.innerHTML = renderAnalysisResult(entityName, resultData);
    } else {
      // Show placeholder for entities without full analysis
      content.innerHTML = `
        <div class="tp-no-data">
          <p>No detailed analysis available for <strong>${entityName}</strong> yet.</p>
          <p>Run a custom analysis on <a href="https://werai.ca/truth-protocol.html" target="_blank">werai.ca</a></p>
        </div>
      `;
    }
  } catch (error) {
    content.innerHTML = `
      <div class="tp-error">
        Error loading analysis: ${error.message}
      </div>
    `;
  }
}

/**
 * Render full analysis result
 */
function renderAnalysisResult(entityName, data) {
  const consensus = data.consensus || {};
  const responses = data.model_responses || [];

  return `
    <div class="tp-analysis">
      <div class="tp-consensus">
        <div class="tp-consensus-label">Consensus</div>
        <div class="tp-consensus-level">${consensus.level || 'Unknown'}</div>
        <div class="tp-consensus-agreement">
          ${consensus.agreement_percentage || 0}% agreement (${responses.length} models)
        </div>
      </div>

      <div class="tp-classification">
        <div class="tp-classification-label">Classification</div>
        <div class="tp-classification-value">${data.classification || 'Unknown'}</div>
      </div>

      ${responses.length > 0 ? `
        <div class="tp-responses">
          <div class="tp-responses-label">Model Analysis</div>
          ${responses.slice(0, 3).map(r => `
            <div class="tp-response">
              <div class="tp-response-model">${r.model}</div>
              <div class="tp-response-classification">${r.classification}</div>
            </div>
          `).join('')}
          ${responses.length > 3 ? `
            <div class="tp-more">+ ${responses.length - 3} more models</div>
          ` : ''}
        </div>
      ` : ''}
    </div>
  `;
}

/**
 * Initialize overlay system
 */
function initOverlay() {
  // Close overlay on click outside
  document.addEventListener('click', (e) => {
    if (activeOverlay && !e.target.closest('.tp-overlay-panel') && !e.target.closest('.tp-indicator')) {
      hideEntityOverlay();
    }
  });
}

// Export
window.TruthProtocol = window.TruthProtocol || {};
window.TruthProtocol.overlay = {
  showHoverCard,
  hideHoverCard,
  showEntityOverlay,
  hideEntityOverlay,
  initOverlay
};

// Make functions available globally for annotator
window.showHoverCard = showHoverCard;
window.hideHoverCard = hideHoverCard;
window.showEntityOverlay = showEntityOverlay;

// Initialize when DOM ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initOverlay);
} else {
  initOverlay();
}

// Run annotation after all scripts loaded
setTimeout(() => {
  if (window.TruthProtocol.annotator) {
    window.TruthProtocol.annotator.processDetectedEntities();
  }
}, 100);
