/**
 * Truth Protocol - Popup Script
 * Extension popup logic
 *
 * WeRAI / HumAIn Integration
 * Patent Pending: USPTO #63900179
 */

const API_BASE = 'https://werai.ca/api/truth';

// Status colors
const STATUS_COLORS = {
  consistent: '#22c55e',
  evolved: '#eab308',
  contradicted: '#f97316',
  memory_hole: '#ef4444',
  unknown: '#6b7280',
  loading: '#3b82f6'
};

// DOM elements
const apiStatusEl = document.getElementById('api-status');
const modelCountEl = document.getElementById('model-count');
const entityListEl = document.getElementById('entity-list');
const toggleDetectionEl = document.getElementById('toggle-detection');
const toggleIndicatorsEl = document.getElementById('toggle-indicators');
const btnRescanEl = document.getElementById('btn-rescan');
const btnAnalyzeEl = document.getElementById('btn-analyze');

/**
 * Initialize popup
 */
async function init() {
  // Load saved settings
  const settings = await chrome.storage.local.get(['detectionEnabled', 'indicatorsEnabled']);
  toggleDetectionEl.checked = settings.detectionEnabled !== false;
  toggleIndicatorsEl.checked = settings.indicatorsEnabled !== false;

  // Check API status
  checkApiStatus();

  // Get entities from current page
  getPageEntities();

  // Set up event listeners
  setupEventListeners();
}

/**
 * Check API health
 */
async function checkApiStatus() {
  try {
    const response = await fetch(`${API_BASE}/health`);
    const data = await response.json();

    if (response.ok) {
      apiStatusEl.textContent = 'Online';
      apiStatusEl.className = 'status-value status-online';
      modelCountEl.textContent = data.total_models || '-';
    } else {
      throw new Error('API error');
    }
  } catch (error) {
    apiStatusEl.textContent = 'Offline';
    apiStatusEl.className = 'status-value status-offline';
    modelCountEl.textContent = '-';
  }
}

/**
 * Get entities detected on current page
 */
async function getPageEntities() {
  try {
    // Query the active tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    // Execute script to get entities
    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: () => {
        // Get all annotated entities
        const wrappers = document.querySelectorAll('.tp-entity-wrapper');
        const entities = new Map();

        wrappers.forEach(wrapper => {
          const name = wrapper.dataset.entity;
          const indicator = wrapper.querySelector('.tp-indicator');
          const status = indicator?.dataset.status || 'unknown';

          if (!entities.has(name)) {
            entities.set(name, { name, status, count: 0 });
          }
          entities.get(name).count++;
        });

        return Array.from(entities.values());
      }
    });

    const entities = results[0]?.result || [];
    renderEntityList(entities);
  } catch (error) {
    console.error('Error getting page entities:', error);
    entityListEl.innerHTML = '<div class="no-entities">Could not scan page</div>';
  }
}

/**
 * Render entity list
 */
function renderEntityList(entities) {
  if (entities.length === 0) {
    entityListEl.innerHTML = '<div class="no-entities">No public figures detected</div>';
    return;
  }

  entityListEl.innerHTML = entities.map(entity => `
    <div class="entity-item" data-entity="${entity.name}">
      <span class="entity-name">${entity.name} (${entity.count})</span>
      <span class="entity-status" style="background: ${STATUS_COLORS[entity.status] || STATUS_COLORS.unknown}"></span>
    </div>
  `).join('');

  // Add click handlers
  entityListEl.querySelectorAll('.entity-item').forEach(item => {
    item.addEventListener('click', () => {
      const name = item.dataset.entity;
      highlightEntityOnPage(name);
    });
  });
}

/**
 * Highlight entity on page
 */
async function highlightEntityOnPage(entityName) {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

  await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: (name) => {
      // Find first instance and scroll to it
      const wrapper = document.querySelector(`.tp-entity-wrapper[data-entity="${name}"]`);
      if (wrapper) {
        wrapper.scrollIntoView({ behavior: 'smooth', block: 'center' });
        wrapper.style.animation = 'tp-highlight 1s ease';
        setTimeout(() => wrapper.style.animation = '', 1000);
      }
    },
    args: [entityName]
  });

  // Close popup
  window.close();
}

/**
 * Set up event listeners
 */
function setupEventListeners() {
  // Toggle detection
  toggleDetectionEl.addEventListener('change', async (e) => {
    await chrome.storage.local.set({ detectionEnabled: e.target.checked });
    sendMessageToTab({ action: 'toggleDetection', enabled: e.target.checked });
  });

  // Toggle indicators
  toggleIndicatorsEl.addEventListener('change', async (e) => {
    await chrome.storage.local.set({ indicatorsEnabled: e.target.checked });
    sendMessageToTab({ action: 'toggleIndicators', enabled: e.target.checked });
  });

  // Rescan button
  btnRescanEl.addEventListener('click', async () => {
    btnRescanEl.textContent = 'Scanning...';
    await sendMessageToTab({ action: 'rescan' });
    setTimeout(() => {
      btnRescanEl.textContent = 'Rescan Page';
      getPageEntities();
    }, 1000);
  });

  // Analyze button
  btnAnalyzeEl.addEventListener('click', () => {
    chrome.tabs.create({ url: 'https://werai.ca/truth-protocol.html' });
  });
}

/**
 * Send message to content script
 */
async function sendMessageToTab(message) {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  try {
    await chrome.tabs.sendMessage(tab.id, message);
  } catch (error) {
    console.error('Error sending message to tab:', error);
  }
}

// Initialize on load
document.addEventListener('DOMContentLoaded', init);
