/**
 * Truth Protocol - Sidebar Script
 * Side panel logic for detailed analysis
 *
 * WeRAI / HumAIn Integration
 * Patent Pending: USPTO #63900179
 */

const API_BASE = 'https://werai.ca/api/truth';

// Status colors
const STATUS_COLORS = {
  consistent: '#22c55e',
  evolved: '#eab308',
  contradicted: '#f97316',
  memory_hole: '#ef4444',
  unknown: '#6b7280'
};

// DOM elements
const statusDot = document.getElementById('status-dot');
const statusText = document.getElementById('status-text');
const inputSubject = document.getElementById('input-subject');
const inputTopic = document.getElementById('input-topic');
const btnAnalyze = document.getElementById('btn-analyze');
const resultsSection = document.getElementById('results-section');
const resultsContainer = document.getElementById('results-container');
const entityContainer = document.getElementById('entity-container');

/**
 * Initialize sidebar
 */
async function init() {
  // Check API status
  await checkApiStatus();

  // Get page entities
  await getPageEntities();

  // Set up event listeners
  setupEventListeners();
}

/**
 * Check API health
 */
async function checkApiStatus() {
  try {
    const response = await fetch(`${API_BASE}/health`);
    const data = await response.json();

    if (response.ok) {
      statusDot.classList.remove('offline');
      statusText.textContent = `Online - ${data.total_models || 0} models`;
    } else {
      throw new Error('API error');
    }
  } catch (error) {
    statusDot.classList.add('offline');
    statusText.textContent = 'Offline';
  }
}

/**
 * Get entities from current page
 */
async function getPageEntities() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: () => {
        const wrappers = document.querySelectorAll('.tp-entity-wrapper');
        const entities = new Map();

        wrappers.forEach(wrapper => {
          const name = wrapper.dataset.entity;
          const indicator = wrapper.querySelector('.tp-indicator');
          const status = indicator?.dataset.status || 'unknown';

          if (!entities.has(name)) {
            entities.set(name, { name, status, count: 0 });
          }
          entities.get(name).count++;
        });

        return Array.from(entities.values());
      }
    });

    const entities = results[0]?.result || [];
    renderEntities(entities);
  } catch (error) {
    console.error('Error getting page entities:', error);
  }
}

/**
 * Render entity list
 */
function renderEntities(entities) {
  if (entities.length === 0) {
    entityContainer.innerHTML = `
      <div class="no-entities">
        <div class="no-entities-icon">?</div>
        <p>No public figures detected on this page</p>
      </div>
    `;
    return;
  }

  entityContainer.innerHTML = entities.map(entity => `
    <div class="entity-card" data-entity="${entity.name}">
      <div class="entity-header">
        <span class="entity-name">${entity.name}</span>
        <span class="entity-status" style="background: ${STATUS_COLORS[entity.status] || STATUS_COLORS.unknown}"></span>
      </div>
      <div class="entity-meta">${getStatusLabel(entity.status)}</div>
      <div class="entity-count">${entity.count} mention${entity.count > 1 ? 's' : ''} on page</div>
    </div>
  `).join('');

  // Add click handlers
  entityContainer.querySelectorAll('.entity-card').forEach(card => {
    card.addEventListener('click', () => {
      const name = card.dataset.entity;
      inputSubject.value = name;
      inputSubject.focus();
    });
  });
}

/**
 * Get status label
 */
function getStatusLabel(status) {
  const labels = {
    consistent: 'Consistent record',
    evolved: 'Position evolved',
    contradicted: 'Contradictions detected',
    memory_hole: 'Previous positions denied',
    unknown: 'Not analyzed',
    loading: 'Checking...',
    error: 'Error checking'
  };
  return labels[status] || 'Unknown';
}

/**
 * Set up event listeners
 */
function setupEventListeners() {
  btnAnalyze.addEventListener('click', runAnalysis);

  // Enter key triggers analysis
  inputSubject.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') runAnalysis();
  });
  inputTopic.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') runAnalysis();
  });
}

/**
 * Run analysis
 */
async function runAnalysis() {
  const subject = inputSubject.value.trim();
  const topic = inputTopic.value.trim();

  if (!subject) {
    inputSubject.focus();
    return;
  }

  // Show loading
  btnAnalyze.disabled = true;
  btnAnalyze.textContent = 'Analyzing...';
  resultsSection.style.display = 'block';
  resultsContainer.innerHTML = `
    <div class="loading">
      <div class="spinner"></div>
      <p>Running multi-model consensus...</p>
      <p style="font-size: 11px; color: #666; margin-top: 8px;">This may take 1-2 minutes</p>
    </div>
  `;

  try {
    const response = await fetch(`${API_BASE}/analyze`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        subject: subject,
        topic: topic || 'general'
      })
    });

    const data = await response.json();
    renderResults(data);
  } catch (error) {
    resultsContainer.innerHTML = `
      <div class="result-card">
        <p style="color: #ef4444;">Analysis failed: ${error.message}</p>
      </div>
    `;
  } finally {
    btnAnalyze.disabled = false;
    btnAnalyze.textContent = 'Run Analysis';
  }
}

/**
 * Render analysis results
 */
function renderResults(data) {
  const consensus = data.consensus || {};
  const responses = data.model_responses || [];

  const classificationColor = STATUS_COLORS[data.classification?.toLowerCase()] || '#f97316';

  resultsContainer.innerHTML = `
    <div class="result-card">
      <div class="result-header">
        <span class="result-subject">${data.subject || 'Unknown'}</span>
        <span class="result-consensus">${consensus.level || 'Unknown'}</span>
      </div>

      <div class="result-classification" style="color: ${classificationColor}">
        ${data.classification || 'Unknown'}
      </div>

      <div style="margin-bottom: 12px;">
        <strong>Topic:</strong> ${data.topic || 'General'}
      </div>

      ${data.summary ? `
        <div style="margin-bottom: 12px; font-size: 13px; line-height: 1.5; color: #b0b0b0;">
          ${data.summary}
        </div>
      ` : ''}

      <div class="result-models">
        ${consensus.agreement_percentage || 0}% agreement across ${responses.length} models
      </div>

      ${responses.length > 0 ? `
        <div style="margin-top: 16px; padding-top: 16px; border-top: 1px solid #ffffff10;">
          <div style="font-size: 11px; color: #888; margin-bottom: 8px;">MODEL RESPONSES</div>
          ${responses.map(r => `
            <div style="display: flex; justify-content: space-between; padding: 6px 0; border-bottom: 1px solid #ffffff05;">
              <span style="font-size: 12px; color: #888;">${r.model}</span>
              <span style="font-size: 12px; color: ${STATUS_COLORS[r.classification?.toLowerCase()] || '#888'};">${r.classification}</span>
            </div>
          `).join('')}
        </div>
      ` : ''}
    </div>
  `;
}

// Initialize on load
document.addEventListener('DOMContentLoaded', init);
